function xopt = genetic(fun, bounds, options)
%GENETIC  simple genetic algorithm.
%
%  XOPT = GENETIC(FUN, BOUNDS, OPTIONS) maximizes the function FUN within
%  the bounds BOUNDS by means of a floating-point genetic algorithm and
%  returns the optimal parameters XOPT.
%
%  BOUNDS(1,:) are the lower, BOUNDS(2,:) the upper bounds of the
%  parameters.
%
%  currently the only genetic operations applied are crossover with a
%  probability given by OPTIONS(1) and mutation with a probability given by
%  OPTIONS(2).  (OPTIONS(3) is reserved for the reproduction
%  probability.) the size of the population is given by OPTIONS(4).
%  OPTIONS(5) is the maximal number of generations.  OPTIONS(6) is the
%  minimum value of the standard deviation to be considered for termination.
%
%  see also INITIALIZE, EVALUATE, SELECTION, MUTATION, CROSSOVER.
%
%  copyleft (C) 1999 by Ernesto Rico-Schmidt <nene@sbox.tu-graz.ac.at>
%  all rights reversed.  see COPYING for more information.
%
%  $Id: genetic.m,v 1.2 2002/08/23 14:18:52 ernesto Exp $

% very simple parameter check
% FIXME: this is to simple
% 
if nargin<3,
  error('please specify all options, I''m not a clairvoyant!');
end

% determine all options
%
Pc = options(1);
Pm = options(2);
popsize = options(4);
maxgen = options(5);
eps = options(6);

% get ready for fun
%
done = 0;
gen = 0;
stats = [];

xopt = bounds(1);
bestf = -inf;

% create start population and evaluate its fitness
%
pop = initialize(popsize, bounds);
fitness = evaluate(fun, pop);

disp('genetic: algorithm statistics');
disp('  gen          max          min         mean          std');

% main loop
%
while ~done,
  gen = gen + 1;

  % select sub-population
  %
  subpop = selection(pop, fitness);

  % perform crossover
  %
  subpop1 = crossover(subpop, Pc);

  % perform mutation
  %
  subpop2 = mutation(subpop, bounds, Pm);
    
  % evaluate fitness
  %
  newpop = [pop; subpop1; subpop2];
  fitness = evaluate(fun, newpop);
  [fitness, index] = sort(fitness);	% sort fitness
  newpop = newpop(index,:);		% sort population

  % select survivors
  %
  newpop = flipud(newpop);		% invert the order
  fitness = flipud(fitness);
  pop = newpop(1:popsize,:);		% select the best
  fitness = fitness(1:popsize,:);
  
  % keep statistics
  %
  [maxf, index] = max(fitness);
  stats = [stats; maxf min(fitness) mean(fitness) std(fitness)];

  if maxf>bestf,
    bestf = maxf;			% update the best fit
    xopt = pop(index(1),:);		% update the best individual
  end

  % show progress
  %
  disp([sprintf('%5.0f %12.6g %12.6g ',gen-1,stats(gen,1),stats(gen,2)), ...
	sprintf('%12.6g %12.6g  ',stats(gen,3), stats(gen,4))]);

  % check for termination
  % FIXME: implement a better convergence test
  %
  if gen==maxgen,
    done = 1;
  elseif (gen>1) & (stats(gen,4)<eps),
    done = 1;
  end
end

% print results: do we got better or only get tired?
%
if gen==maxgen,
  disp('genetic: algorithm reached maximun number of generations.');
else
  disp('genetic: algorithm succesfully converged.');
end
